package com.yiboshi.science.rest.v1;

import com.yiboshi.arch.base.ResponseDataModel;
import com.yiboshi.science.base.Pagination;
import com.yiboshi.science.config.annotation.Logs;
import com.yiboshi.science.config.security.SecurityUserHolder;
import com.yiboshi.science.entity.ComProjectCheck;
import com.yiboshi.science.entity.ComProjectAudit;
import com.yiboshi.science.entity.SelectListItem;
import com.yiboshi.science.enumeration.CommonEnum;
import com.yiboshi.science.param.dto.ComProjectCheckDTO;
import com.yiboshi.science.param.dto.DataStatisticsDTO;
import com.yiboshi.science.param.query.ComProjectCheckQueryVO;
import com.yiboshi.science.rest.BaseController;
import com.yiboshi.science.service.ComProjectCheckService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * 功能：中期考核表 接口
 *
 * @author lkl
 * @version 2021-08-26
 */
@Api(tags = "com-project-check", description = "中期考核表")
@RestController
@RequestMapping("/v1/science-admin/com-project-check")
public class ComProjectCheckController extends BaseController<ComProjectCheckService, ComProjectCheckQueryVO, ComProjectCheckDTO, ComProjectCheck> {
    @Autowired
    private ComProjectCheckService comProjectCheckService;

    /**
     * 功能:分页查询
     *
     * @param vo 查询条件
     */
    @ApiOperation(value = "1.02 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getListByPage")
    public ResponseDataModel<Pagination<ComProjectCheckDTO>> getListByPage(@Validated ComProjectCheckQueryVO vo, BindingResult bindingResult) {
        Pagination<ComProjectCheckDTO> page = comProjectCheckService.getListByPage(vo);
        page.getDataList().forEach((e) -> {
            if (null != e.getProjState())
                e.setProjStateName(CommonEnum.projState.getEnum(e.getProjState()).getDescription());
            if (null != e.getCheckState())
                e.setCheckStateName(CommonEnum.checkState.getEnum(e.getCheckState()).getDescription());
        });
        return ResponseDataModel.ok(page);
    }

    /**
     * 获取统计数据
     */
    @ApiOperation(value = "获取统计数据", httpMethod = "GET", notes = "获取统计数据")
    @GetMapping
    @RequestMapping("/getCount")
    public ResponseDataModel<DataStatisticsDTO> getCount(ComProjectCheck e) {
        ComProjectCheckQueryVO vo=new ComProjectCheckQueryVO();
        vo.setCheckYear(e.getCheckYear());
        return ResponseDataModel.ok(comProjectCheckService.getCount(vo));
    }

    /**
     * 保存中期考核表信息并上报
     *
     * @param dto
     * @return
     */
    @ApiOperation(value = "保存中期考核表信息并上报", httpMethod = "POST", notes = "保存中期考核表信息并上报")
    @PostMapping
    @RequestMapping("/saveAndReport")
    public ResponseDataModel<String> saveAndReport(@RequestBody ComProjectCheckDTO dto) {
        return ResponseDataModel.ok(comProjectCheckService.saveAndReport(dto, SecurityUserHolder.getUnitId(), SecurityUserHolder.getUnitCode()));
    }

    /**
     * 保存中期考核表信息
     *
     * @param dto
     * @return
     */
    @ApiOperation(value = "保存", httpMethod = "POST", notes = "保存")
    @PostMapping
    @RequestMapping("/save")
//    @SystemLogs(value = CommonEnum.LogType.ProjectCheckSave)
    public ResponseDataModel<String> save(@RequestBody ComProjectCheckDTO dto) {
        return ResponseDataModel.ok(comProjectCheckService.save(dto));
    }

    /**
     * 根据id获取项目信息
     *
     * @param id 查询条件
     */
    @ApiOperation(value = "根据id获取项目信息", httpMethod = "GET", notes = "根据id获取项目信息")
    @GetMapping
    @RequestMapping("/getCheckInfoById")
    public ResponseDataModel<ComProjectCheckDTO> getCheckInfoById(String id) {
        return ResponseDataModel.ok(comProjectCheckService.getCheckInfoById(id));
    }

    /**
     * 项目/任务书/中期考核表审核
     */
    @ApiOperation(value = "项目/任务书/中期考核表审核", httpMethod = "POST", notes = "项目/任务书/中期考核表审核")
    @PostMapping
    @RequestMapping("/audit")
    @Logs(value = CommonEnum.logType.checkAudit)
    public ResponseDataModel<String> audit(@RequestBody ComProjectAudit comProjectAudit) {
        comProjectCheckService.audit(comProjectAudit, SecurityUserHolder.getUnitId(), SecurityUserHolder.getUnitCode());
        return ResponseDataModel.ok("审核成功");
    }

    @ApiOperation(value = "获取任务书状态", httpMethod = "GET", notes = "获取任务书状态")
    @GetMapping
    @RequestMapping("/getCheckStateList")
    public ResponseDataModel<List<SelectListItem>> getCheckStateList() {
        List<SelectListItem> list = new ArrayList<>(CommonEnum.checkState.values().length);
        for (CommonEnum.checkState type : CommonEnum.checkState.values()) {
            SelectListItem item = new SelectListItem();
            item.setKey(type.getCode().toString());
            item.setTitle(type.getDescription());
            list.add(item);
        }
        return ResponseDataModel.ok(list);
    }
}
