package com.yiboshi.science.rest.v1;

import com.yiboshi.arch.base.ResponseDataModel;
import com.yiboshi.science.base.Pagination;
import com.yiboshi.science.config.annotation.Logs;
import com.yiboshi.science.entity.ComExpert;
import com.yiboshi.science.entity.SystemUser;
import com.yiboshi.science.enumeration.CommonEnum;
import com.yiboshi.science.param.dto.ComPersonDTO;
import com.yiboshi.science.param.dto.SystemUserDTO;
import com.yiboshi.science.param.query.SystemUserQueryVO;
import com.yiboshi.science.rest.BaseController;
import com.yiboshi.science.service.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 功能：用户信息表 接口
 *
 * @author lkl
 * @version 2021-08-26
 */
@Api(tags = "system-user", description = "用户信息表")
@RestController
@RequestMapping("/v1/science-admin/system-user")
public class SystemUserController extends BaseController<SystemUserService, SystemUserQueryVO, SystemUserDTO, SystemUser> {

    @Autowired
    private SystemUserService systemUserService;

    @Autowired
    private SystemUserRoleService systemUserRoleService;

    @Autowired
    private ComExpertService comExpertService;

    /**
     * 功能:分页查询
     *
     * @param vo 查询条件
     */
    @ApiOperation(value = "1.02 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getListByPage")
    @PreAuthorize("hasAnyRole('ADMIN')")
    public ResponseDataModel<Pagination<SystemUserDTO>> getListByPage(@Validated SystemUserQueryVO vo, BindingResult bindingResult) {
        Pagination<SystemUserDTO> page = systemUserService.getListByPage(vo);
        if (null != page && null != page.getDataList() && page.getDataList().size() != 0) {
            page.getDataList().forEach((e) -> {
                e.setStateName(CommonEnum.userState.getEnum(e.getNoteState()).getDescription());
            });
        }
        return ResponseDataModel.ok(page);
    }

    /**
     * 功能:修改人员状态
     *
     * @param e 输入值
     */
    @ApiOperation(value = "1.02", httpMethod = "POST", notes = "1.02 修改人员状态")
    @RequestMapping("/updateState")
    @PostMapping
    @Logs(value = CommonEnum.logType.userStateUpdate)
    public ResponseDataModel<String> updateState(@RequestBody SystemUser e) {
        return ResponseDataModel.ok(systemUserService.updateStateById(e.getId(), e.getNoteState()));
    }

    @ApiOperation(value = "重置密码", httpMethod = "POST", notes = "重置密码")
    @RequestMapping("/resetUserPwd")
    @PostMapping
    @Logs(value = CommonEnum.logType.userReset)
    public ResponseDataModel<String> resetUserPwd(@Validated @RequestBody ComPersonDTO dto) {
        return ResponseDataModel.ok(systemUserService.resetPwd(dto));
    }

    @ApiOperation(value = "通过Id获取用户信息", httpMethod = "GET", notes = "通过Id获取用户信息")
    @GetMapping
    @RequestMapping("/getSystemUserById")
    @PreAuthorize("hasAnyRole('ADMIN')")
    public ResponseDataModel<SystemUserDTO> getSystemUserById(String id) {
        SystemUserDTO systemUserDTO = systemUserService.getSystemUserById(id);
        systemUserDTO.setRole(systemUserRoleService.getRolesListByUserId(id));
        return ResponseDataModel.ok(systemUserDTO);
    }

    @ApiOperation(value = "更新系统用户信息", httpMethod = "POST", notes = "更新系统用户信息")
    @RequestMapping("/updateSystemUser")
    @PostMapping
    @PreAuthorize("hasAnyRole('ADMIN')")
    public ResponseDataModel<String> updateSystemUser(@Validated @RequestBody SystemUserDTO dto) {
        SystemUser user = systemUserService.getById(dto.getId());
        String r = StringUtils.join(dto.getRole().toArray(), ",");
        List<String> roles = systemUserRoleService.getRolesListByUserId(dto.getId());
        for (String e : roles) {
            if (!r.contains(e)) {
                systemUserRoleService.deleteRoleByUserId(dto.getId(), e);
            }
        }
        String s = StringUtils.join(roles.toArray(), ",");
        for (String e : dto.getRole()) {
            if (!s.contains(e)) {
                systemUserRoleService.addRoleByUserId(dto.getId(), e);
                if (e.equals(CommonEnum.systemRole.expert.getCode().toString())) {
                    ComExpert comExpert = new ComExpert();
                    comExpert.setPersonId(user.getPersonId());
                    comExpert = comExpertService.getEntity(comExpert);
                    if (null == comExpert) {
                        comExpert = new ComExpert();
                        comExpert.setPersonId(user.getPersonId());
                        comExpert.setExpertState(1);
                        comExpert.setReportState(1);
                        comExpertService.insert(comExpert);
                    }
                }
            }
        }
        return ResponseDataModel.ok(systemUserService.updateSystemUser(dto));
    }
}
