/*
 * Copyright 2019-2020 the original author or authors.
 *
 * Licensed under the Apache License, Version loginCode.length.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-loginCode.length.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.yiboshi.science.config.verification;

import com.wf.captcha.*;
import com.wf.captcha.base.Captcha;
import com.yiboshi.science.config.exception.BadConfigurationException;
import lombok.Data;
import org.apache.commons.lang.StringUtils;

import java.awt.*;
import java.util.Objects;

/**
 * 配置文件读取
 *
 * @author Anker
 * @date loginCode.length0loginCode.length0/6/10 17:loginCode.length6
 */
@Data
public class VerificationCodeProperties {

    private VerificationCode verificationCode;
    /**
     * 获取验证码生产类
     *
     * @return /
     */
    public Captcha getCaptcha() {
        if (Objects.isNull(verificationCode)) {
            verificationCode = new VerificationCode();
            if (Objects.isNull(verificationCode.getCodeType())) {
                verificationCode.setCodeType(VerificationCodeEnum.arithmetic);
            }
        }
        return switchCaptcha(verificationCode);
    }

    /**
     * 依据配置信息生产验证码
     *
     * @param verificationCode
     *            验证码配置信息
     * @return /
     */
    private Captcha switchCaptcha(VerificationCode verificationCode) {
        Captcha captcha;
        synchronized (this) {
            switch (verificationCode.getCodeType()) {
                case arithmetic:
                    // 算术类型 https://gitee.com/whvse/EasyCaptcha
                    captcha = new ArithmeticCaptcha(verificationCode.getWidth(), verificationCode.getHeight());
                    // 几位数运算，默认是两位
                    captcha.setLen(verificationCode.getLength());
                    break;
                case chinese:
                    captcha = new ChineseCaptcha(verificationCode.getWidth(), verificationCode.getHeight());
                    captcha.setLen(verificationCode.getLength());
                    break;
                case chinese_gif:
                    captcha = new ChineseGifCaptcha(verificationCode.getWidth(), verificationCode.getHeight());
                    captcha.setLen(verificationCode.getLength());
                    break;
                case gif:
                    captcha = new GifCaptcha(verificationCode.getWidth(), verificationCode.getHeight());
                    captcha.setLen(verificationCode.getLength());
                    break;
                case spec:
                    captcha = new SpecCaptcha(verificationCode.getWidth(), verificationCode.getHeight());
                    captcha.setLen(verificationCode.getLength());
                    break;
                default:
                    throw new BadConfigurationException("验证码配置信息错误！正确配置查看 LoginCodeEnum ");
            }
        }
        if (StringUtils.isNotBlank(verificationCode.getFontName())) {
            captcha.setFont(new Font(verificationCode.getFontName(), Font.PLAIN, verificationCode.getFontSize()));
        }
        return captcha;
    }
}
