package com.yiboshi.science.rest.v1;

import com.yiboshi.arch.base.ResponseDataModel;
import com.yiboshi.science.base.Pagination;
import com.yiboshi.science.config.annotation.Logs;
import com.yiboshi.science.entity.ComBatch;
import com.yiboshi.science.entity.SelectListItem;
import com.yiboshi.science.enumeration.CommonEnum;
import com.yiboshi.science.param.dto.ComBatchDTO;
import com.yiboshi.science.param.query.ComBatchQueryVO;
import com.yiboshi.science.rest.BaseController;
import com.yiboshi.science.service.ComBatchService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.*;

/**
 * 功能：附件目录表 接口
 *
 * @author lkl
 * @version 2021-08-26
 */
@Api(tags = "com-batch", description = "批次表")
@RestController
@RequestMapping("/v1/science-admin/com-batch")
public class ComBatchController extends BaseController<ComBatchService, ComBatchQueryVO, ComBatchDTO, ComBatch> {
    @Autowired
    private ComBatchService comBatchService;

    @ApiOperation(value = "分页查询", httpMethod = "GET", notes = "根据参数获取列表")
    @GetMapping
    @RequestMapping("/getListByPage")
    public ResponseDataModel<Pagination<ComBatchDTO>> getListByPage(@Validated ComBatchQueryVO vo, BindingResult bindingResult) {
        Pagination<ComBatchDTO> page = comBatchService.getListByPage(vo);
        return ResponseDataModel.ok(page);
    }

    @ApiOperation(value = "通过Id获取批次", httpMethod = "GET", notes = "通过Id获取批次")
    @GetMapping
    @RequestMapping("/getBatchById")
    public ResponseDataModel<ComBatchDTO> getBatchById(String id) {
        return ResponseDataModel.ok(comBatchService.getBatchById(id));
    }

    @ApiOperation(value = "年度批次添加", httpMethod = "POST", notes = "年度批次添加")
    @RequestMapping("/addBatch")
    @PostMapping
    @Logs(value = CommonEnum.logType.batchAdd)
    public ResponseDataModel<String> addBatch(@Validated @RequestBody ComBatchDTO dto) {
        return ResponseDataModel.ok(comBatchService.addBatch(dto));
    }

    @ApiOperation(value = "年度批次修改", httpMethod = "POST", notes = "年度批次修改")
    @RequestMapping("/updateBatch")
    @PostMapping
    @Logs(value = CommonEnum.logType.batchUpdate)
    public ResponseDataModel<String> updateBatch(@Validated @RequestBody ComBatchDTO dto) {
        return ResponseDataModel.ok(comBatchService.updateBatch(dto));
    }

    @ApiOperation(value = "删除批次", httpMethod = "DELETE", notes = "删除单位")
    @DeleteMapping(value = "deleteBatch/{id}")
    @Logs(value = CommonEnum.logType.batchDelete)
    public ResponseDataModel<String> deleteBatch(@PathVariable String id) {
        return ResponseDataModel.ok(comBatchService.deleteBatch(id));
    }

    @ApiOperation(value = "获取当前申报或审核时间", httpMethod = "GET", notes = "获取当前申报或审核时间")
    @GetMapping
    @RequestMapping("/getCurrentYearBatch")
    public ResponseDataModel<ComBatchDTO> getCurrentYearBatch(int type, Integer systemType, Integer timeType) {
        return ResponseDataModel.ok(comBatchService.getCurrentYearBatch(type, systemType, timeType));
    }

    /**
     * 功能:获取申报年度下拉列表
     */
    @ApiOperation(value = "获取申报年度下拉列表", httpMethod = "GET", notes = "获取申报年度下拉列表")
    @GetMapping
    @RequestMapping("/getCurrentReportYearList")
    public ResponseDataModel<Map<String, Object>> getCurrentReportYearList(Integer systemType) {
        ComBatchDTO comBatchDTO = comBatchService.getCurrentBatch(systemType, null);
        Integer year = (null != comBatchDTO && Objects.nonNull(comBatchDTO.getYear())) ? comBatchDTO.getYear() : Calendar.getInstance().get(Calendar.YEAR);
        List<SelectListItem> list = new ArrayList<>();
        for (Integer i = year; i > year - 10; i--) {
            SelectListItem item = new SelectListItem();
            item.setTitle(i + "年");
            item.setKey(i.toString());
            list.add(item);
        }
        Map<String, Object> data = new HashMap<String, Object>(2) {
            {
                put("value", year);
                put("list", list);
            }
        };
        return ResponseDataModel.ok(data);
    }

    @ApiOperation(value = "获取申报年度列表", httpMethod = "GET", notes = "获取申报年度列表")
    @GetMapping
    @RequestMapping("/getBatchList")
    public ResponseDataModel<List<ComBatchDTO>> getBatchList(@Validated ComBatch entity) {
        return ResponseDataModel.ok(comBatchService.getBatchList(entity));
    }

}
