package com.yiboshi.science.rest.v1;

import com.itextpdf.text.DocumentException;
import com.yiboshi.arch.base.ResponseDataModel;
import com.yiboshi.science.base.Pagination;
import com.yiboshi.science.config.annotation.Logs;
import com.yiboshi.science.config.security.SecurityUserHolder;
import com.yiboshi.science.entity.ComProjectAudit;
import com.yiboshi.science.entity.ComProjectTask;
import com.yiboshi.science.entity.SelectListItem;
import com.yiboshi.science.entity.SystemParameter;
import com.yiboshi.science.enumeration.CommonEnum;
import com.yiboshi.science.param.dto.ComFileDTO;
import com.yiboshi.science.param.dto.ComProjectTaskDTO;
import com.yiboshi.science.param.dto.DataStatisticsDTO;
import com.yiboshi.science.param.query.ComProjectTaskQueryVO;
import com.yiboshi.science.rest.BaseController;
import com.yiboshi.science.service.ComProjectTaskService;
import com.yiboshi.science.service.SystemParameterService;
import com.yiboshi.science.utils.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.ServletOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 功能：任务书表 接口
 *
 * @author lkl
 * @version 2021-08-26
 */
@Api(tags = "com-project-test", description = "任务书表")
@RestController
@RequestMapping("/v1/science-admin/com-project-task")
public class ComProjectTaskController extends BaseController<ComProjectTaskService, ComProjectTaskQueryVO, ComProjectTaskDTO, ComProjectTask> {

    @Autowired
    private ComProjectTaskService ComProjectTaskService;

    @Autowired
    private  SystemParameterService systemParameterService;


    /**
     * 功能:分页查询
     *
     * @param vo 查询条件
     */
    @ApiOperation(value = "1.02 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getListByPage")
    public ResponseDataModel<Pagination<ComProjectTaskDTO>> getListByPage(@Validated ComProjectTaskQueryVO vo, BindingResult bindingResult) {
        if (!StringUtil.isContainsRole(SecurityUserHolder.getRoles(), CommonEnum.systemRole.sys.getCode().toString())) {
            vo.setAppPersonId(SecurityUserHolder.getPersonId());
        }
        vo.setProjState(CommonEnum.projState.pass.getCode());
        Pagination<ComProjectTaskDTO> page = ComProjectTaskService.getListByPage(vo);
        if (null != page && null != page.getDataList() && page.getDataList().size() != 0) {
            page.getDataList().forEach((e) -> {
                e.setTaskStateName(CommonEnum.taskState.getEnum(e.getTaskState()).getDescription());
            });
        }
        return ResponseDataModel.ok(page);
    }

    /**
     * 获取统计数据
     */
    @ApiOperation(value = "获取统计数据", httpMethod = "GET", notes = "获取统计数据")
    @GetMapping
    @RequestMapping("/getCount")
    public ResponseDataModel<DataStatisticsDTO> getCount(ComProjectTaskQueryVO vo) {
        vo.setAppPersonId(SecurityUserHolder.getPersonId());
        vo.setProjState(CommonEnum.projState.pass.getCode());
        vo.setReportYear(vo.getReportYear());
        return ResponseDataModel.ok(ComProjectTaskService.getCount(vo));
    }

    /**
     * 根据id获取任务书信息
     *
     * @param id
     */
    @ApiOperation(value = "根据id获取项目信息", httpMethod = "GET", notes = "根据id获取项目信息")
    @GetMapping
    @RequestMapping("/getTaskByProjId")
    public ResponseDataModel<ComProjectTaskDTO> getTaskByProjId(String id) {
        return ResponseDataModel.ok(ComProjectTaskService.getTaskByProjId(id));
    }

    /**
     * 保存任务书信息
     *
     * @param dto
     * @return
     */
    @ApiOperation(value = "保存任务书信息", httpMethod = "POST", notes = "保存任务书信息")
    @PostMapping
    @RequestMapping("/save")
    @Logs(value = CommonEnum.logType.taskSave)
    public ResponseDataModel<String> save(@RequestBody ComProjectTaskDTO dto) {
        return ResponseDataModel.ok(ComProjectTaskService.save(dto));
    }

    /**
     * 项目/任务书/中期考核表上报
     */
    @ApiOperation(value = "项目/任务书/中期考核表上报", httpMethod = "POST", notes = "项目/任务书/中期考核表上报")
    @PostMapping
    @RequestMapping("/report")
    @Logs(value = CommonEnum.logType.taskReport)
    public ResponseDataModel<String> report(@RequestBody ComProjectAudit comProjectAudit) {
        ComProjectTaskService.report(comProjectAudit, SecurityUserHolder.getUnitId(), SecurityUserHolder.getUnitCode());
        return ResponseDataModel.ok("上报成功");
    }

    /**
     * 项目/任务书/中期考核表审核
     */
    @ApiOperation(value = "项目/任务书/中期考核表审核", httpMethod = "POST", notes = "项目/任务书/中期考核表审核")
    @PostMapping
    @RequestMapping("/audit")
    @Logs(value = CommonEnum.logType.taskAudit)
    public ResponseDataModel<String> audit(@RequestBody ComProjectAudit comProjectAudit) {
        ComProjectTaskService.audit(comProjectAudit, SecurityUserHolder.getUnitId(), SecurityUserHolder.getUnitCode());
        return ResponseDataModel.ok("审核成功");
    }

    @ApiOperation(value = "获取任务书状态", httpMethod = "GET", notes = "获取任务书状态")
    @GetMapping
    @RequestMapping("/getTaskStateList")
    public ResponseDataModel<List<SelectListItem>> getTaskStateList() {
        List<SelectListItem> list = new ArrayList<>(CommonEnum.taskState.values().length);
        for (CommonEnum.taskState type : CommonEnum.taskState.values()) {
            SelectListItem item = new SelectListItem();
            item.setKey(type.getCode().toString());
            item.setTitle(type.getDescription());
            list.add(item);
        }
        return ResponseDataModel.ok(list);
    }

    @ApiOperation(value = "项目导出", httpMethod = "POST", notes = "项目导出")
    @RequestMapping("/export/{id}")
    @PostMapping
    public void export(@PathVariable String id) throws DocumentException, IOException {
        try {
            ComProjectTaskDTO dto = ComProjectTaskService.getTaskByProjId(id);
            List<SystemParameter> list = systemParameterService.getListByType(67);
//          WordUtils.getHtmlContent(dto, "C:\\Users\\kylin\\Desktop\\导出Pdf\\output.docx");
            // 生成PDF文件字节数组
            byte[] pdfBytes = TaskInfoToPDFUtil.createContractPdf(dto,list);
//          byte[] pdfBytes = TaskToPDF.createContractPdf(dto);
//          byte[] pdfBytes = TalentToPDF.createContractPdf(dto);
            // 设置响应头
            response.setContentType("application/pdf");
            String fileName = new String((dto.getProjName() + ".pdf").getBytes("UTF-8"), "ISO-8859-1");
            response.setHeader("Content-Disposition", "attachment; filename=" + fileName);
            response.setContentLength(pdfBytes.length);
            // 写入响应流
            try (ServletOutputStream out = response.getOutputStream()) {
                out.write(pdfBytes);
                out.flush();
            }
        } catch (Exception e) {
            response.setContentType("text/plain;charset=UTF-8");
            response.getWriter().write("导出PDF失败：" + e.getMessage());
        }
    }

}
