package com.yiboshi.science.rest.v1;

import com.yiboshi.arch.base.ResponseDataModel;
import com.yiboshi.science.base.Pagination;
import com.yiboshi.science.config.security.SecurityUserHolder;
import com.yiboshi.science.entity.ComProjectGroup;
import com.yiboshi.science.enumeration.CommonEnum;
import com.yiboshi.science.param.dto.*;
import com.yiboshi.science.param.query.ComProjectAuditQueryVO;
import com.yiboshi.science.param.query.ComProjectGroupQueryVO;
import com.yiboshi.science.rest.BaseController;
import com.yiboshi.science.service.ComProjectGroupService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

@Api(tags = "com-project-group", description = "项目分组表")
@RestController
@RequestMapping("/v1/science-admin/com-project-group")
public class ComProjectGroupController extends BaseController<ComProjectGroupService, ComProjectGroupQueryVO, ComProjectGroupDTO, ComProjectGroup> {
    @Autowired
    private ComProjectGroupService comProjectGroupService;

    @ApiOperation(value = "1.02 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getProjectGroupListByPage")
    public ResponseDataModel<Pagination<ComProjectGroupDTO>> getProjectGroupListByPage(@Validated ComProjectGroupQueryVO vo, BindingResult bindingResult) {
        Pagination<ComProjectGroupDTO> page = comProjectGroupService.getProjectGroupListByPage(vo);
        return ResponseDataModel.ok(page);
    }

    @ApiOperation(value = "根据id获取项目组", httpMethod = "GET", notes = "根据id获取项目组")
    @GetMapping
    @RequestMapping("/getGroupById")
    public ResponseDataModel<ComProjectGroupDTO> getGroupById(@Validated String id) {
        return ResponseDataModel.ok(comProjectGroupService.getGroupById(id));
    }

    @ApiOperation(value = "新增项目组", httpMethod = "POST", notes = "新增项目组", consumes = MediaType.APPLICATION_JSON_VALUE)
    @PostMapping
    @RequestMapping("/addGroup")
    public ResponseDataModel<String> addGroup(@Validated @RequestBody ComProjectGroupDTO E) {
        return ResponseDataModel.ok(comProjectGroupService.save(E));
    }

    @ApiOperation(value = "修改项目组", httpMethod = "PUT", notes = "2.02 修改数据", consumes = MediaType.APPLICATION_JSON_VALUE)
    @ApiImplicitParam(dataType = "string", name = "id", value = "主键", example = "1", type = "path")
    @PutMapping(value = "/updateGroup")
    public ResponseDataModel<String> updateGroup(@Validated @RequestBody ComProjectGroupDTO D) {
        return ResponseDataModel.ok(comProjectGroupService.save(D));
    }

    @ApiOperation(value = "删除项目组", httpMethod = "DELETE", notes = "删除项目组")
    @DeleteMapping
    @RequestMapping(value = "deleteByGroupId/{id}")
    public ResponseDataModel<String> deleteByGroupId(@Validated @PathVariable String id) {
        return ResponseDataModel.ok(comProjectGroupService.deleteByGroupId(id));
    }

    @ApiOperation(value = "项目组明细调整", httpMethod = "GET", notes = "项目组明细调整")
    @GetMapping
    @RequestMapping("/updataProjectGroupAdjust")
    public ResponseDataModel<String> updataProjectGroupAdjust(@RequestParam String Ids, @RequestParam String groupId) {
        String[] split = Ids.split(",");
        List<String> IdList = Arrays.asList(split);
        if (comProjectGroupService.updataProjectGroupAdjust(IdList, groupId))
            return ResponseDataModel.ok("数据保存成功！");
        else
            return ResponseDataModel.ok("项目组已经分配有专家，不能进行项目调整！");
    }

    @ApiOperation(value = "项目组专家分配", httpMethod = "GET", notes = "项目组专家分配")
    @GetMapping
    @RequestMapping("/assignProjectGroup")
    public ResponseDataModel<String> assignProjectGroup(@RequestParam int systemType, @RequestParam String groupIds, @RequestParam String expertIds) {
        String[] split = groupIds.split(",");
        List<String> GroupList = Arrays.asList(split);

        split = expertIds.split(",");
        List<String> ExpertList = Arrays.asList(split);

        return ResponseDataModel.ok(comProjectGroupService.assignProjectGroup(systemType, GroupList, ExpertList));
    }

    @ApiOperation(value = "删除分配项目组专家及项目专家关系", httpMethod = "DELETE", notes = "删除分配项目组专家及项目专家关系")
    @DeleteMapping(value = "deleteAssignGroupExpert/{id}")
    public ResponseDataModel<String> deleteAssignGroupExpert(@PathVariable String id) {
        return ResponseDataModel.ok(comProjectGroupService.deleteAssignGroupExpert(id));
    }

    @ApiOperation(value = "项目插入分组", httpMethod = "GET", notes = "项目插入分组")
    @GetMapping
    @RequestMapping("/InsertProjectGroup")
    public ResponseDataModel<String> InsertProjectGroup(@RequestParam String Ids, @RequestParam String groupId) {
        String[] split = Ids.split(",");
        List<String> IdList = Arrays.asList(split);
        if (comProjectGroupService.InsertProjectGroup(IdList, groupId))
            return ResponseDataModel.ok("项目分组成功！");
        else
            return ResponseDataModel.ok("项目分组失败，请检查项目分组是否存在！");
    }

    @ApiOperation(value = "项目组移除项目", httpMethod = "GET", notes = "项目组删除项目")
    @GetMapping
    @RequestMapping("/deleteGroupProjectExpert")
    public ResponseDataModel<String> deleteGroupProjectExpert(@RequestParam String groupId, @RequestParam String projId) {
        if (comProjectGroupService.deleteGroupProjectExpert(groupId, projId))
            return ResponseDataModel.ok("项目组移除项目成功！");
        else
            return ResponseDataModel.ok("项目组移除项目失败，请检查项目是否分配了评审专家！");
    }

    @ApiOperation(value = "项目二级学科统计", httpMethod = "GET", notes = "项目二级学科统计")
    @GetMapping
    @RequestMapping("/getProjectKnowledgeStatisticByGroupId")
    public ResponseDataModel<List<ComProjectKnowledgeStatisticDTO>> getProjectKnowledgeStatisticByGroupId(@RequestParam String groupId) {
        List<ComProjectKnowledgeStatisticDTO> list = comProjectGroupService.getProjectKnowledgeStatisticByGroupId(groupId);
        return ResponseDataModel.ok(list);
    }

    @ApiOperation(value = "项目组添加二级学科下所属项目", httpMethod = "GET", notes = "项目组添加二级学科下所属项目")
    @GetMapping
    @RequestMapping("/addProjectGroupKnowledge")
    @PreAuthorize("hasAnyRole('GOV')")
    public ResponseDataModel<String> addProjectGroupKnowledge(@RequestParam String groupId, @RequestParam String knowledgeIds, @RequestParam Integer reportYear) {
        String[] split = knowledgeIds.split(",");
        List<String> KnowledgeList = Arrays.asList(split);

        ComProjectAuditQueryVO v = new ComProjectAuditQueryVO();
        v.setAuditUnitId(SecurityUserHolder.getUnitId());
        v.setAuditMethod(CommonEnum.auditMethod.last.getCode());
        v.setReportYear(reportYear);
        if (comProjectGroupService.addProjectGroupKnowledge(groupId, KnowledgeList, v))
            return ResponseDataModel.ok("数据保存成功！");
        else
            return ResponseDataModel.ok("数据保存失败，请检查！");
    }

    @ApiOperation(value = "项目组移除二级学科下所属项目", httpMethod = "GET", notes = "项目组移除二级学科下所属项目")
    @GetMapping
    @RequestMapping("/deleteProjectGroupKnowledge")
    public ResponseDataModel<String> deleteProjectGroupKnowledge(@RequestParam String groupId, @RequestParam String knowledgeId) {
        if (comProjectGroupService.deleteProjectGroupKnowledge(groupId, knowledgeId))
            return ResponseDataModel.ok("项目组移除二级学科下所属项目成功！");
        else
            return ResponseDataModel.ok("项目组移除二级学科下所属项目失败，请检查！");
    }

    @ApiOperation(value = "根据学科Id和项目组Id移除项目", httpMethod = "GET", notes = "根据学科Id和项目组Id移除项目")
    @GetMapping
    @RequestMapping("/deleteProjectGroupDetail")
    public ResponseDataModel<String> deleteProjectGroupDetail(@RequestParam String groupId, @RequestParam String knowledgeId) {
        if (comProjectGroupService.deleteProjectGroupDetail(groupId, knowledgeId))
            return ResponseDataModel.ok("项目组移除项目成功！");
        else
            return ResponseDataModel.ok("项目组移除项目失败，请检查！");
    }

    @ApiOperation(value = "项目信息", httpMethod = "GET", notes = "项目信息")
    @GetMapping
    @RequestMapping("/getGroupProjectInfo")
    public ResponseDataModel<Map<String, Object>> getGroupProjectInfo(@RequestParam String groupIds) {
        String[] split = groupIds.split(",");
        List<String> groupList = Arrays.asList(split);
        return ResponseDataModel.ok(comProjectGroupService.getGroupProjectInfo(groupList));
    }
}
