package com.yiboshi.science.rest.v1;

import com.yiboshi.arch.base.ResponseDataModel;
import com.yiboshi.science.base.Pagination;
import com.yiboshi.science.config.annotation.Logs;
import com.yiboshi.science.config.security.SecurityUserHolder;
import com.yiboshi.science.entity.ComProjectAssign;
import com.yiboshi.science.enumeration.CommonEnum;
import com.yiboshi.science.param.dto.ComProjectAssignDTO;
import com.yiboshi.science.param.query.ComProjectAssignQueryVO;
import com.yiboshi.science.rest.BaseController;
import com.yiboshi.science.service.ComExpertSpecService;
import com.yiboshi.science.service.ComProjectAssignService;
import com.yiboshi.science.service.SystemParameterService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

import static com.yiboshi.science.utils.StringUtil.hideAllPhoneNum;
import static com.yiboshi.science.utils.StringUtil.hideAllIdCardNum;

@Api(tags = "com-project-assign", description = "项目分配表")
@RestController
@RequestMapping("/v1/science-admin/com-project-assign")
public class ComProjectAssignController extends BaseController<ComProjectAssignService, ComProjectAssignQueryVO, ComProjectAssignDTO, ComProjectAssign> {

    @Autowired
    private ComProjectAssignService comProjectAssignService;

    @Autowired
    private ComExpertSpecService comExpertSpecService;

    @Autowired
    private SystemParameterService systemParameterService;

    @ApiOperation(value = "1.02 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getListByPage")
    public ResponseDataModel<Pagination<ComProjectAssignDTO>> getListByPage(@Validated ComProjectAssignQueryVO vo, BindingResult bindingResult) {
        return getPaginationResponseDataModel(vo);
    }

    @ApiOperation(value = "1.02 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getAuditListByPage")
    public ResponseDataModel<Pagination<ComProjectAssignDTO>> getAuditListByPage(@Validated ComProjectAssignQueryVO vo, BindingResult bindingResult) {
        vo.setExpertId(SecurityUserHolder.getExpertId());
        return getPaginationResponseDataModel(vo);
    }

    private ResponseDataModel<Pagination<ComProjectAssignDTO>> getPaginationResponseDataModel(@Validated ComProjectAssignQueryVO vo) {
        Pagination<ComProjectAssignDTO> page = comProjectAssignService.getListByPage(vo);
        if (null != page && null != page.getDataList() && page.getDataList().size() != 0) {
            page.getDataList().forEach((e) -> {
                e.setTalentCategoryName(systemParameterService.getParaName(e.getTalentCategory()));
                e.setCertId(hideAllIdCardNum(e.getCertId()));
                e.setStateName(e.getAuditState().equals(2) ? "已评审" : e.getAuditState().equals(1) ? "未提交" : "未评审");
            });
        }
        return ResponseDataModel.ok(page);
    }

    @ApiOperation(value = "1.02", httpMethod = "POST", notes = "项目分配")
    @RequestMapping("/assignProject")
    @PostMapping
    @Logs(value = CommonEnum.logType.projectAssign)
    public ResponseDataModel<String> assignProject(@RequestBody Map<String, Object> map) {
        return ResponseDataModel.ok(comProjectAssignService.assignProject(map));
    }

    @ApiOperation(value = "根据Id专家评分信息", httpMethod = "GET", notes = "根据Id专家评分信息")
    @GetMapping
    @RequestMapping("/getAssignExpertById")
    public ResponseDataModel<ComProjectAssignDTO> getAssignExpertById(String id) {
        ComProjectAssignDTO dto = comProjectAssignService.getAssignExpertById(id);
        if (null != dto) {
            dto.setCertId(hideAllIdCardNum(dto.getCertId()));
            dto.setMobile(hideAllPhoneNum(dto.getMobile()));
        }
        return ResponseDataModel.ok(dto);
    }

    @ApiOperation(value = "更新专家评分记录", httpMethod = "POST", notes = "更新专家评分记录")
    @RequestMapping("/expertEvaluation")
    @PostMapping
    public ResponseDataModel<String> expertEvaluation(@Validated @RequestBody ComProjectAssignDTO dto) {
        return ResponseDataModel.ok(comProjectAssignService.expertEvaluation(dto));
    }

    @ApiOperation(value = "删除评分记录", httpMethod = "DELETE", notes = "删除评分记录")
    @DeleteMapping(value = "deleteAssignExpert/{id}")
    @Logs(value = CommonEnum.logType.assignDelete)
    public ResponseDataModel<String> deleteAssignExpert(@PathVariable String id) {
        return ResponseDataModel.ok(comProjectAssignService.deleteAssignExpert(id));
    }
}
