package com.yiboshi.science.rest.v1;

import com.yiboshi.arch.base.ResponseDataModel;
import com.yiboshi.science.base.Pagination;
import com.yiboshi.science.entity.ComTalentGroup;
import com.yiboshi.science.param.dto.ComTalentGroupDTO;
import com.yiboshi.science.param.dto.ComTalentKnowledgeStatisticDTO;
import com.yiboshi.science.param.query.ComTalentGroupQueryVO;
import com.yiboshi.science.rest.BaseController;
import com.yiboshi.science.service.ComTalentGroupService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

@Api(tags = "com-talent-group", description = "人才分组表")
@RestController
@RequestMapping("/v1/science-admin/com-talent-group")
public class ComTalentGroupController extends BaseController<ComTalentGroupService, ComTalentGroupQueryVO, ComTalentGroupDTO, ComTalentGroup> {

    @Autowired
    private ComTalentGroupService comTalentGroupService;

    @ApiOperation(value = "1.02 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getTalentGroupListByPage")
    public ResponseDataModel<Pagination<ComTalentGroupDTO>> getTalentGroupListByPage(@Validated ComTalentGroupQueryVO vo, BindingResult bindingResult) {
        Pagination<ComTalentGroupDTO> page = comTalentGroupService.getTalentGroupListByPage(vo);
        return ResponseDataModel.ok(page);
    }

    @ApiOperation(value = "根据id获取人才分组", httpMethod = "GET", notes = "根据id获取人才分组")
    @GetMapping
    @RequestMapping("/getGroupById")
    public ResponseDataModel<ComTalentGroupDTO> getGroupById(@Validated String id) {
        return ResponseDataModel.ok(comTalentGroupService.getGroupById(id));
    }

    @ApiOperation(value = "新增人才分组", httpMethod = "POST", notes = "新增人才分组", consumes = MediaType.APPLICATION_JSON_VALUE)
    @PostMapping
    @RequestMapping("/addGroup")
    public ResponseDataModel<String> addGroup(@Validated @RequestBody ComTalentGroupDTO E) {
        return ResponseDataModel.ok(comTalentGroupService.save(E));
    }

    @ApiOperation(value = "修改人才分组", httpMethod = "PUT", notes = "2.02 修改数据", consumes = MediaType.APPLICATION_JSON_VALUE)
    @ApiImplicitParam(dataType = "string", name = "id", value = "主键", example = "1", type = "path")
    @PutMapping(value = "/updateGroup")
    public ResponseDataModel<String> updateGroup(@Validated @RequestBody ComTalentGroupDTO D) {
        return ResponseDataModel.ok(comTalentGroupService.save(D));
    }

    @ApiOperation(value = "删除人才分组", httpMethod = "DELETE", notes = "删除人才分组")
    @DeleteMapping
    @RequestMapping(value = "deleteByGroupId/{id}")
    public ResponseDataModel<String> deleteByGroupId(@Validated @PathVariable String id) {
        return ResponseDataModel.ok(comTalentGroupService.deleteByGroupId(id));
    }

    @ApiOperation(value = "人才分组明细调整", httpMethod = "GET", notes = "人才分组明细调整")
    @GetMapping
    @RequestMapping("/updataTalentGroupAdjust")
    public ResponseDataModel<String> updataTalentGroupAdjust(@RequestParam String Ids, @RequestParam String groupId) {
        String[] split = Ids.split(",");
        List<String> IdList = Arrays.asList(split);
        if (comTalentGroupService.updataTalentGroupAdjust(IdList, groupId))
            return ResponseDataModel.ok("数据保存成功！");
        else
            return ResponseDataModel.ok("人才分组已经分配有专家，不能进行人才分组调整！");
    }

    @ApiOperation(value = "人才分组专家分配", httpMethod = "GET", notes = "人才分组专家分配")
    @GetMapping
    @RequestMapping("/assignTalentGroup")
    public ResponseDataModel<String> assignTalentGroup(@RequestParam int systemType, @RequestParam String groupIds, @RequestParam String expertIds) {
        String[] split = groupIds.split(",");
        List<String> GroupList = Arrays.asList(split);

        split = expertIds.split(",");
        List<String> ExpertList = Arrays.asList(split);

        return ResponseDataModel.ok(comTalentGroupService.assignTalentGroup(systemType, GroupList, ExpertList));
    }

    @ApiOperation(value = "删除分配人才分组专家及专家关系", httpMethod = "DELETE", notes = "删除分配人才分组专家及专家关系")
    @DeleteMapping(value = "deleteAssignGroupExpert/{id}")
    public ResponseDataModel<String> deleteAssignGroupExpert(@PathVariable String id) {
        return ResponseDataModel.ok(comTalentGroupService.deleteAssignGroupExpert(id));
    }

    @ApiOperation(value = "人次插入分组", httpMethod = "GET", notes = "人才插入分组")
    @GetMapping
    @RequestMapping("/InsertTalentGroup")
    public ResponseDataModel<String> InsertTalentGroup(@RequestParam String Ids, @RequestParam String groupId) {
        String[] split = Ids.split(",");
        List<String> IdList = Arrays.asList(split);
        if (comTalentGroupService.InsertTalentGroup(IdList, groupId))
            return ResponseDataModel.ok("人才分组成功！");
        else
            return ResponseDataModel.ok("人才分组失败，请检查人才分组是否存在！");
    }

    @ApiOperation(value = "人才分组移除人才", httpMethod = "GET", notes = "人才分组删除人才")
    @GetMapping
    @RequestMapping("/deleteGroupTalentExpert")
    public ResponseDataModel<String> deleteGroupTalentExpert(@RequestParam String groupId, @RequestParam String talentId) {
        if (comTalentGroupService.deleteGroupTalentExpert(groupId, talentId))
            return ResponseDataModel.ok("人才分组移除人才成功！");
        else
            return ResponseDataModel.ok("人才分组移除人才失败，请检查人才是否分配了评审专家！");
    }

    @ApiOperation(value = "人才二级学科统计", httpMethod = "GET", notes = "人才二级学科统计")
    @GetMapping
    @RequestMapping("/getTalentKnowledgeStatisticByGroupId")
    public ResponseDataModel<List<ComTalentKnowledgeStatisticDTO>> getTalentKnowledgeStatisticByGroupId(@RequestParam String groupId) {
        List<ComTalentKnowledgeStatisticDTO> list = comTalentGroupService.getTalentKnowledgeStatisticByGroupId(groupId);
        return ResponseDataModel.ok(list);
    }

    @ApiOperation(value = "人才信息", httpMethod = "GET", notes = "人才信息")
    @GetMapping
    @RequestMapping("/getGroupTalentInfo")
    public ResponseDataModel<Map<String, Object>> getGroupTalentInfo(@RequestParam String groupIds) {
        String[] split = groupIds.split(",");
        List<String> groupList = Arrays.asList(split);
        return ResponseDataModel.ok(comTalentGroupService.getGroupTalentInfo(groupList));
    }
}
