package com.yiboshi.science.rest.v1;

import com.yiboshi.arch.base.ResponseDataModel;
import com.yiboshi.science.base.Pagination;
import com.yiboshi.science.config.annotation.Logs;
import com.yiboshi.science.config.bean.SystemProperties;
import com.yiboshi.science.config.security.SecurityUserHolder;
import com.yiboshi.science.entity.ComProjectAudit;
import com.yiboshi.science.entity.ComProjectConclusion;
import com.yiboshi.science.enumeration.CommonEnum;
import com.yiboshi.science.param.dto.ComProjectConclusionDTO;
import com.yiboshi.science.param.dto.ComUnitDTO;
import com.yiboshi.science.param.dto.DataStatisticsDTO;
import com.yiboshi.science.param.query.ComProjectConclusionQueryVO;
import com.yiboshi.science.rest.BaseController;
import com.yiboshi.science.service.ComProjectConclusionService;
import com.yiboshi.science.service.ComUnitService;
import com.yiboshi.science.utils.StringUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;

/**
 * 功能：结题信息表 接口
 * @author lkl
 * @version 2021-08-26
 */
@Api(tags = "com-project-conclusion",description = "结题信息表")
@RestController
@RequestMapping("/v1/science-admin/com-project-conclusion")
public class ComProjectConclusionController extends BaseController<ComProjectConclusionService, ComProjectConclusionQueryVO, ComProjectConclusionDTO, ComProjectConclusion> {

    @Autowired
    private ComProjectConclusionService comProjectConclusionService;

    @Autowired
    private ComUnitService comUnitService;

    private final SystemProperties properties;

    public ComProjectConclusionController(SystemProperties properties) {
        this.properties = properties;
    }

    /**
     * 功能:分页查询
     *
     * @param vo 查询条件
     */
    @ApiOperation(value = "分页查询", httpMethod = "GET", notes = "根据参数获取列表")
    @GetMapping
    @RequestMapping("/getListByPage")
    @PreAuthorize("hasAnyRole(\"REPORT\",\"ADMIN\")")
    public ResponseDataModel<Pagination<ComProjectConclusionDTO>> getListByPage(@Validated ComProjectConclusionQueryVO vo, BindingResult bindingResult) {
        if (!StringUtil.isContainsRole(SecurityUserHolder.getRoles(), CommonEnum.systemRole.sys.getCode().toString())) {
            vo.setAppPersonId(SecurityUserHolder.getPersonId());
        }
        return getPaginationResponseDataModel(vo);
    }

    /**
     * 获取统计数据
     */
    @ApiOperation(value = "获取统计数据", httpMethod = "GET", notes = "获取统计数据")
    @GetMapping
    @RequestMapping("/getCount")
    public ResponseDataModel<DataStatisticsDTO> getCount(@Validated ComProjectConclusionDTO vo) {
        vo.setAppPersonId(SecurityUserHolder.getPersonId());
        return ResponseDataModel.ok(comProjectConclusionService.getCount(vo));
    }

    /**
     * 功能:分页查询
     *
     * @param vo 查询条件
     */
    @ApiOperation(value = "分页查询", httpMethod = "GET", notes = "根据参数获取列表")
    @GetMapping
    @RequestMapping("/getAuditListByPage")
    @PreAuthorize("hasAnyRole(\"GOV\")")
    public ResponseDataModel<Pagination<ComProjectConclusionDTO>> getAuditListByPage(@Validated ComProjectConclusionQueryVO vo, BindingResult bindingResult) {
        return getPaginationResponseDataModel(vo);
    }


    private ResponseDataModel<Pagination<ComProjectConclusionDTO>> getPaginationResponseDataModel(@Validated ComProjectConclusionQueryVO vo) {
        Pagination<ComProjectConclusionDTO> page = comProjectConclusionService.getListByPage(vo);
        if (null != page && null != page.getDataList() && page.getDataList().size() != 0) {
            page.getDataList().forEach((e) -> {
                if (null != e.getProjState())
                    e.setStateName(CommonEnum.conclusionState.getEnum(e.getState()).getDescription());
            });
        }
        return ResponseDataModel.ok(page);
    }
    /**
     * 根据id获取结题信息
     *
     * @param id 查询条件
     */
    @ApiOperation(value = "根据id获取结题信息", httpMethod = "GET", notes = "根据id获取结题信息")
    @GetMapping
    @RequestMapping("/getConclusionByProjId")
    public ResponseDataModel<ComProjectConclusionDTO> getConclusionByProjId(@Validated String id) {
        return ResponseDataModel.ok(comProjectConclusionService.getByProjId(id));
    }
    /**
     * 保存项目结题信息
     *
     * @param comProjectConclusionDTO
     */
    @ApiOperation(value = "保存项目结题信息", httpMethod = "POST", notes = "保存项目结题信息")
    @PostMapping
    @RequestMapping("/save")
    @Logs(value = CommonEnum.logType.conclusionSave)
    public ResponseDataModel<String> save(@Validated @RequestBody ComProjectConclusionDTO comProjectConclusionDTO, BindingResult bindingResult) {
        return ResponseDataModel.ok(comProjectConclusionService.save(comProjectConclusionDTO));
    }
    /**
     * 上传论文
     *
     * @param comProjectConclusionDTO
     */
    @ApiOperation(value = "上传论文", httpMethod = "POST", notes = "上传论文")
    @PostMapping
    @RequestMapping("/saveThesis")
    @Logs(value = CommonEnum.logType.conclusionSave)
    public ResponseDataModel<String> saveThesis(@Validated @RequestBody ComProjectConclusionDTO comProjectConclusionDTO, BindingResult bindingResult) {
        return ResponseDataModel.ok(comProjectConclusionService.saveThesis(comProjectConclusionDTO,"ef2123fc-e7c0-4521-be59-244f9052253b"));
    }
    /**
     * 项目结题上报
     *
     * @param
     */
    @ApiOperation(value = "项目结题", httpMethod = "PUT", notes = "项目结题")
    @PostMapping
    @RequestMapping("/report")
    @Logs(value = CommonEnum.logType.conclusionReport)
    public ResponseDataModel<String> report(@Validated @RequestBody ComProjectConclusion e) {

        String treeCode = SecurityUserHolder.getUnitCode();
        // 获取上级单位
        treeCode = treeCode.substring(0, properties.getDefaultCodeLength());
        ComUnitDTO unit = comUnitService.getByTreeCode(treeCode);

        return ResponseDataModel.ok(comProjectConclusionService.report(e.getProjId(),unit.getId(),treeCode));
    }

    @ApiOperation(value = "论文审核", httpMethod = "POST", notes = "论文审核")
    @PostMapping
    @RequestMapping("/thesisAudit")
    @Logs(value = CommonEnum.logType.thesisAudit)
    public ResponseDataModel<String> thesisAudit(@Validated @RequestBody ComProjectAudit comProjectAudit, BindingResult bindingResult) {
        comProjectConclusionService.thesisAudit(comProjectAudit);
        return ResponseDataModel.ok("审核成功");
    }

    @ApiOperation(value = "结题", httpMethod = "POST", notes = "结题")
    @RequestMapping(value = "createConclusionRecord/{id}")
    @PostMapping
    public ResponseDataModel<String> createConclusionRecord(@PathVariable String id){
        comProjectConclusionService.createConclusionRecord(id,"ef2123fc-e7c0-4521-be59-244f9052253b");
        return ResponseDataModel.ok("提交成功");
    }

    @RequestMapping("/getPdfById")
    @GetMapping
    public void getPdfById(@Validated String id)throws IOException{
        comProjectConclusionService.exportPdfById(response,id);
    }
}
