package com.yiboshi.science.rest.v1;

import com.yiboshi.arch.base.ResponseDataModel;
import com.yiboshi.science.base.Pagination;
import com.yiboshi.science.config.annotation.Logs;
import com.yiboshi.science.config.security.SecurityUserHolder;
import com.yiboshi.science.entity.ComTalentAssign;
import com.yiboshi.science.enumeration.CommonEnum;
import com.yiboshi.science.param.dto.ComTalentAssignDTO;
import com.yiboshi.science.param.query.ComTalentAssignQueryVO;
import com.yiboshi.science.rest.BaseController;
import com.yiboshi.science.service.ComExpertSpecService;
import com.yiboshi.science.service.ComTalentAssignService;
import com.yiboshi.science.service.SystemParameterService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

import static com.yiboshi.science.utils.StringUtil.hideAllIdCardNum;
import static com.yiboshi.science.utils.StringUtil.hideAllPhoneNum;

@Api(tags = "com-talent-assign", description = "人才分配表")
@RestController
@RequestMapping("/v1/science-admin/com-talent-assign")
public class ComTalentAssignController extends BaseController<ComTalentAssignService, ComTalentAssignQueryVO, ComTalentAssignDTO, ComTalentAssign> {

    @Autowired
    private ComTalentAssignService comTalentAssignService;

    @Autowired
    private ComExpertSpecService comExpertSpecService;

    @Autowired
    private SystemParameterService systemParameterService;

    @ApiOperation(value = "1.02 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getListByPage")
    public ResponseDataModel<Pagination<ComTalentAssignDTO>> getListByPage(@Validated ComTalentAssignQueryVO vo, BindingResult bindingResult) {
        return getPaginationResponseDataModel(vo);
    }

    @ApiOperation(value = "1.02 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getAuditListByPage")
    public ResponseDataModel<Pagination<ComTalentAssignDTO>> getAuditListByPage(@Validated ComTalentAssignQueryVO vo, BindingResult bindingResult) {
        vo.setExpertId(SecurityUserHolder.getExpertId());
        return getPaginationResponseDataModel(vo);
    }


    private ResponseDataModel<Pagination<ComTalentAssignDTO>> getPaginationResponseDataModel(@Validated ComTalentAssignQueryVO vo) {
        Pagination<ComTalentAssignDTO> page = comTalentAssignService.getListByPage(vo);
        if (null != page && null != page.getDataList() && page.getDataList().size() != 0) {
            page.getDataList().forEach((e) -> {
                e.setSpecList(comExpertSpecService.getListByExpertId(e.getExpertId()));
                e.setCertId(hideAllIdCardNum(e.getCertId()));
                e.setStateName(e.getAuditState().equals(2) ? "已评审" : e.getAuditState().equals(1) ? "未提交" : "未评审");
                e.setTalentCategoryName(systemParameterService.getParaName(e.getTalentCategory()));
                e.setTitleName(systemParameterService.getParaName(e.getTitle()));
                e.setProfessionName(systemParameterService.getParaName(e.getProfession()));
                e.setExpertTitleName(systemParameterService.getParaName(e.getExpertTitle()));
            });
        }
        return ResponseDataModel.ok(page);
    }



    @ApiOperation(value = "1.02", httpMethod = "POST", notes = "项目分配")
    @RequestMapping("/assignTalent")
    @PostMapping
    @Logs(value = CommonEnum.logType.talentAssign)
    public ResponseDataModel<String> assignTalent(@RequestBody Map<String, Object> map) {
        return ResponseDataModel.ok(comTalentAssignService.assignTalent(map));
    }

    @ApiOperation(value = "根据Id专家评分信息", httpMethod = "GET", notes = "根据Id专家评分信息")
    @GetMapping
    @RequestMapping("/getTalentAssignInfoById")
    public ResponseDataModel<ComTalentAssignDTO> getTalentAssignInfoById(String id) {
        ComTalentAssignDTO dto = comTalentAssignService.getAssignExpertById(id);
        return ResponseDataModel.ok(dto);
    }

    @ApiOperation(value = "更新专家评分记录", httpMethod = "POST", notes = "更新专家评分记录")
    @RequestMapping("/expertEvaluation")
    @PostMapping
    public ResponseDataModel<String> expertEvaluation(@Validated @RequestBody ComTalentAssignDTO dto) {
        return ResponseDataModel.ok(comTalentAssignService.expertEvaluation(dto));
    }

    @ApiOperation(value = "删除评分记录", httpMethod = "DELETE", notes = "删除评分记录")
    @DeleteMapping(value = "deleteAssignExpert/{id}")
    @Logs(value = CommonEnum.logType.assignDelete)
    public ResponseDataModel<String> deleteAssignExpert(@PathVariable String id) {
        return ResponseDataModel.ok(comTalentAssignService.deleteAssignExpert(id));
    }
}
