package com.yiboshi.science.rest.v1;

import com.yiboshi.arch.base.ResponseDataModel;
import com.yiboshi.arch.exception.BusinessException;
import com.yiboshi.science.base.Pagination;
import com.yiboshi.science.config.annotation.Logs;
import com.yiboshi.science.config.bean.SystemProperties;
import com.yiboshi.science.config.security.SecurityUserHolder;
import com.yiboshi.science.entity.ComProjectAudit;
import com.yiboshi.science.entity.ComUnit;
import com.yiboshi.science.enumeration.CommonEnum;
import com.yiboshi.science.param.dto.*;
import com.yiboshi.science.param.query.ComProjectAuditQueryVO;
import com.yiboshi.science.rest.BaseController;
import com.yiboshi.science.service.ComProjectAssignService;
import com.yiboshi.science.service.ComProjectAuditService;
import com.yiboshi.science.service.ComUnitService;
import com.yiboshi.science.service.SystemParameterService;
import com.yiboshi.science.utils.IDCardUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.Objects;

import static com.yiboshi.science.utils.StringUtil.hideAllIdCardNum;
import static com.yiboshi.science.utils.StringUtil.hideAllPhoneNum;

/**
 * 功能：审核表 接口
 *
 * @author lkl
 * @version 2021-08-26
 */
@Api(tags = "com-project-report", description = "上报表")
@RestController
@RequestMapping("/v1/science-admin/com_project_audit")
public class ComProjectAuditController extends BaseController<ComProjectAuditService, ComProjectAuditQueryVO, ComProjectAuditDTO, ComProjectAudit> {

    @Autowired
    private ComProjectAuditService comProjectAuditService;

    @Autowired
    private ComProjectAssignService comProjectAssignService;

    private final SystemProperties properties;

    @Autowired
    private SystemParameterService systemParameterService;

    @Autowired
    private ComUnitService comUnitService;

    public ComProjectAuditController(SystemProperties properties) {
        this.properties = properties;
    }

    /**
     * 功能:分页查询
     *
     * @param vo 查询条件
     */
    @ApiOperation(value = "分页查询", httpMethod = "GET", notes = "根据参数获取列表")
    @GetMapping
    @RequestMapping(value = "/getListByPage")
    @PreAuthorize("hasAnyRole('ADMIN')")
    public ResponseDataModel<Pagination<ComProjectAuditDTO>> getListByPage(@Validated ComProjectAuditQueryVO vo, BindingResult bindingResult) {
        Pagination<ComProjectAuditDTO> page = comProjectAuditService.getListByPage(vo);
        if (null != page && null != page.getDataList() && page.getDataList().size() != 0) {
            page.getDataList().forEach((e) -> {
                if (Objects.nonNull(e.getAuditType()))
                    e.setAuditTypeName(CommonEnum.auditType.getEnum(e.getAuditType()).getDescription());
                if (Objects.nonNull(e.getAuditMethod()))
                    e.setAuditMethodName(CommonEnum.auditMethod.getEnum(e.getAuditMethod()).getDescription());
                if (Objects.nonNull(e.getAuditResult()))
                    e.setAuditResultName(CommonEnum.auditResult.getEnum(e.getAuditResult()).getDescription());
            });
        }
        return ResponseDataModel.ok(page);
    }

    /**
     * 修改/添加评审记录
     *
     * @param comProjectAuditDTO
     */
    @ApiOperation(value = "修改/添加评审记录", httpMethod = "POST", notes = "修改/添加评审记录")
    @PostMapping
    @RequestMapping("/save")
    @PreAuthorize("hasAnyRole('ADMIN')")
    @Logs(value = CommonEnum.logType.auditRecordSave)
    public ResponseDataModel<String> save(@RequestBody ComProjectAuditDTO comProjectAuditDTO, BindingResult bindingResult) {
        if (Objects.nonNull(comProjectAuditDTO.getAuditUnitId())) {
            ComUnit unit = comUnitService.getById(comProjectAuditDTO.getAuditUnitId());
            if (null == unit)
                throw new BusinessException("审核单位不存在！");
            comProjectAuditDTO.setUnitLevel(unit.getTreeCode().length() / properties.getDefaultCodeLength());
        } else
            throw new BusinessException("审核单位不存在！");
        return ResponseDataModel.ok(comProjectAuditService.save(comProjectAuditDTO));
    }

    /**
     * 删除评审记录
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "删除评审记录", httpMethod = "DELETE", notes = "删除评审记录")
    @DeleteMapping
    @RequestMapping(value = "deleteRecord/{id}")
    @Logs(value = CommonEnum.logType.auditRecordDelete)
    public ResponseDataModel<String> deleteRecord(@PathVariable String id) {
        return ResponseDataModel.ok(comProjectAuditService.deleteRecord(id));
    }

    /**
     * 功能:分页查询
     *
     * @param vo 查询条件
     */
    @ApiOperation(value = "1.03 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getAuditListByPage")
    @PreAuthorize("hasAnyRole('UNIT','GOV')")
    public ResponseDataModel<Pagination<ComProjectAuditDTO>> getAuditListByPage(@Validated ComProjectAuditQueryVO vo, BindingResult bindingResult) {
        vo.setAuditUnitId(SecurityUserHolder.getUnitId());
        return getPaginationResponseDataModel(vo);
    }

    private ResponseDataModel<Pagination<ComProjectAuditDTO>> getPaginationResponseDataModel(@Validated ComProjectAuditQueryVO vo) {
        Pagination<ComProjectAuditDTO> page = comProjectAuditService.getAuditListByPage(vo);
        if (null != page && null != page.getDataList() && page.getDataList().size() != 0) {
            page.getDataList().forEach((e) -> {
                if (vo.getPageIndex() != -1) {
                    e.setMobile(hideAllPhoneNum(e.getMobile()));
                    e.setCertId(hideAllIdCardNum(e.getCertId()));
                }
                if (null != e.getAuditResult())
                    e.setAuditResultName(CommonEnum.auditResult.getEnum(e.getAuditResult()).getDescription());
                e.setTalentCategoryName(systemParameterService.getParaName(e.getTalentCategory()));
                e.setTitleName(systemParameterService.getParaName(e.getTitle()));
                e.setProfessionName(systemParameterService.getParaName(e.getProfession()));
                if (vo.getAuditMethod().equals(CommonEnum.auditMethod.last.getCode()) && vo.getAuditType().equals(CommonEnum.auditType.project.getCode())) {
//                  e.setAssignList(comProjectAssignService.getAssignExpertList(e.getAuditObjectId()));
                    e.setAge(IDCardUtil.getAge(e.getCertId()));
                    BigDecimal total = new BigDecimal(0);
                    BigDecimal gov = new BigDecimal(0);
                    BigDecimal self = new BigDecimal(0);
                    if (Objects.nonNull(e.getTotalFunding())) {
                        total = e.getTotalFunding();
                    }
                    if (Objects.nonNull(e.getGovFunding())) {
                        gov = e.getGovFunding();
                    }
                    self = total.subtract(gov);
                    e.setSelfFunding(self);
                }
            });
        }
        return ResponseDataModel.ok(page);
    }

    /**
     * 获取统计数据
     */
    @ApiOperation(value = "获取统计数据", httpMethod = "GET", notes = "获取统计数据")
    @GetMapping
    @RequestMapping("/getCount")
    @PreAuthorize("hasAnyRole('UNIT','GOV')")
    public ResponseDataModel<DataStatisticsDTO> getCount(ComProjectAuditQueryVO vo) {
        vo.setAuditUnitId(SecurityUserHolder.getUnitId());
        return ResponseDataModel.ok(comProjectAuditService.getCount(vo));
    }

    /**
     * 获取统计数据
     */
    @ApiOperation(value = "获取统计数据", httpMethod = "GET", notes = "获取统计数据")
    @GetMapping
    @RequestMapping("/getConclusionAuditCount")
    @PreAuthorize("hasAnyRole('UNIT','GOV')")
    public ResponseDataModel<DataStatisticsDTO> getConclusionAuditCount(ComProjectAuditQueryVO vo) {
        vo.setAuditUnitId(SecurityUserHolder.getUnitId());
        vo.setAuditMethod(CommonEnum.auditMethod.audit.getCode());
        return ResponseDataModel.ok(comProjectAuditService.getConclusionAuditCount(vo));
    }

    /**
     * 获取统计数据
     */
    @ApiOperation(value = "获取统计数据", httpMethod = "GET", notes = "获取统计数据")
    @GetMapping
    @RequestMapping("/getTalentAuditCount")
    @PreAuthorize("hasAnyRole('UNIT','GOV')")
    public ResponseDataModel<DataStatisticsDTO> getTalentAuditCount(ComProjectAuditQueryVO vo) {
        vo.setAuditUnitId(SecurityUserHolder.getUnitId());
        return ResponseDataModel.ok(comProjectAuditService.getTalentAuditCount(vo));
    }

    /**
     * 功能:分页查询
     *
     * @param vo 查询条件
     */
    @ApiOperation(value = "1.03 分页查询", httpMethod = "GET", notes = "1.02 根据参数获取列表")
    @GetMapping
    @RequestMapping("/getStatisticListByPage")
    @PreAuthorize("hasAnyRole('UNIT','GOV')")
    public ResponseDataModel<Pagination<ComProjectAuditDTO>> getStatisticListByPage(@Validated ComProjectAuditQueryVO vo, BindingResult bindingResult) {
        vo.setAuditUnitId(SecurityUserHolder.getUnitId());
        vo.setAuditMethod(CommonEnum.auditMethod.audit.getCode());
        Pagination<ComProjectAuditDTO> page = comProjectAuditService.getStatisticListByPage(vo);
        if (null != page && null != page.getDataList() && page.getDataList().size() != 0) {
            page.getDataList().forEach((e) -> {
                if (null != e.getAuditResult())
                    e.setAuditResultName(CommonEnum.auditResult.getEnum(e.getAuditResult()).getDescription());
                if (null != e.getProjState())
                    e.setProjStateName(CommonEnum.projState.getEnum(e.getProjState()).getDescription());
            });
        }
        return ResponseDataModel.ok(page);
    }

    @ApiOperation(value = "项目分配列表", httpMethod = "GET", notes = "项目分配列表")
    @GetMapping
    @RequestMapping("/getAssignProjectListByPage")
    public ResponseDataModel<Pagination<ComProjectAuditDTO>> getAssignProjectListByPage(@Validated ComProjectAuditQueryVO vo, BindingResult bindingResult) {
        vo.setAuditUnitId(SecurityUserHolder.getUnitId());
        vo.setSystemType(SecurityUserHolder.projectType());
        vo.setAuditType(1);
        vo.setAuditMethod(CommonEnum.auditMethod.last.getCode());
        return ResponseDataModel.ok(comProjectAuditService.getAuditListByPage(vo));
    }

    @ApiOperation(value = "获取未分配的项目列表", httpMethod = "GET", notes = "获取未分配的项目列表")
    @GetMapping
    @RequestMapping("/getUnAssignProjectListByPage")
    public ResponseDataModel<Pagination<ComProjectAuditDTO>> getUnAssignProjectListByPage(@Validated ComProjectAuditQueryVO vo, BindingResult bindingResult) {
        vo.setAuditUnitId(SecurityUserHolder.getUnitId());
        vo.setKnowledgeId(null);
        vo.setAuditType(1);
        vo.setAuditMethod(CommonEnum.auditMethod.last.getCode());
        return ResponseDataModel.ok(comProjectAuditService.getUnAssignProjectListByPage(vo));
    }

    @ApiOperation(value = "获取未分配的人才列表", httpMethod = "GET", notes = "获取未分配的人才列表")
    @GetMapping
    @RequestMapping("/getUnAssignTalentListByPage")
    public ResponseDataModel<Pagination<ComProjectAuditDTO>> getUnAssignTalentListByPage(@Validated ComProjectAuditQueryVO vo, BindingResult bindingResult) {
        vo.setAuditUnitId(SecurityUserHolder.getUnitId());
        vo.setKnowledgeId(null);
        vo.setAuditType(6);
        vo.setAuditMethod(CommonEnum.auditMethod.last.getCode());
        return ResponseDataModel.ok(comProjectAuditService.getUnAssignTalentListByPage(vo));
    }
}
